<?php
// auth.php - Authentication API
require_once 'config.php';
require 'cors.php';

$pdo = getConnection();
$method = $_SERVER['REQUEST_METHOD'];
$input = json_decode(file_get_contents('php://input'), true);

switch($method) {
    case 'POST':
        $action = $_GET['action'] ?? '';
        
        if ($action === 'register') {
            register($pdo, $input);
        } elseif ($action === 'login') {
            login($pdo, $input);
        } else {
            sendError('Invalid action');
        }
        break;
        
    default:
        sendError('Method not allowed', 405);
}

function register($pdo, $data) {
    // Validate input
    if (!isset($data['name'], $data['email'], $data['phone'], $data['password'])) {
        sendError('Missing required fields');
    }
    
    $name = trim($data['name']);
    $email = trim($data['email']);
    $phone = trim($data['phone']);
    $password = $data['password'];
    $role = $data['role'] ?? 'driver';
    
    if (empty($name) || empty($email) || empty($phone) || empty($password)) {
        sendError('All fields are required');
    }
    
    // Check if email already exists
    $stmt = $pdo->prepare("SELECT id FROM users WHERE email = ?");
    $stmt->execute([$email]);
    if ($stmt->fetch()) {
        sendError('Email already registered');
    }
    
    // Hash password
    $hashedPassword = password_hash($password, PASSWORD_DEFAULT);
    
    // Insert user
    try {
        $stmt = $pdo->prepare("INSERT INTO users (name, email, phone, password, role) VALUES (?, ?, ?, ?, ?)");
        $stmt->execute([$name, $email, $phone, $hashedPassword, $role]);
        
        $userId = $pdo->lastInsertId();
        
        // If mechanic, create mechanic profile
        if ($role === 'mechanic' && isset($data['garage_name'], $data['latitude'], $data['longitude'])) {
            $stmt = $pdo->prepare("INSERT INTO mechanics (user_id, garage_name, latitude, longitude, address, services) VALUES (?, ?, ?, ?, ?, ?)");
            $stmt->execute([
                $userId,
                $data['garage_name'],
                $data['latitude'],
                $data['longitude'],
                $data['address'] ?? '',
                $data['services'] ?? ''
            ]);
        }
        
        sendResponse([
            'success' => true,
            'message' => 'User registered successfully',
            'user_id' => $userId
        ]);
        
    } catch(PDOException $e) {
        sendError('Registration failed: ' . $e->getMessage());
    }
}

function login($pdo, $data) {
    // Validate input
    if (!isset($data['email'], $data['password'])) {
        sendError('Email and password required');
    }
    
    $email = trim($data['email']);
    $password = $data['password'];
    
    // Find user
    $stmt = $pdo->prepare("SELECT id, name, email, phone, password, role FROM users WHERE email = ?");
    $stmt->execute([$email]);
    $user = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$user || !password_verify($password, $user['password'])) {
        sendError('Invalid email or password');
    }
    
    // Remove password from response
    unset($user['password']);
    
    // Get mechanic details if user is mechanic
    if ($user['role'] === 'mechanic') {
        $stmt = $pdo->prepare("SELECT * FROM mechanics WHERE user_id = ?");
        $stmt->execute([$user['id']]);
        $mechanic = $stmt->fetch(PDO::FETCH_ASSOC);
        $user['mechanic_details'] = $mechanic;
    }
    
    sendResponse([
        'success' => true,
        'message' => 'Login successful',
        'user' => $user
    ]);
}
?>